/* 
 * Google Maps Leads Extractor - Background Service Worker
 * Extracts business data from Google Maps search results
 */

// Constants
const API_URL = 'https://api.leads-extractor.com/v2/findBusinesses';

// All Google Maps domains (must match manifest.json host_permissions)
const GOOGLE_MAPS_DOMAINS = [
    'https://www.google.com/*',
    'https://www.google.ac/*',
    'https://www.google.ad/*',
    'https://www.google.ae/*',
    'https://www.google.com.af/*',
    'https://www.google.com.ag/*',
    'https://www.google.com.ai/*',
    'https://www.google.al/*',
    'https://www.google.am/*',
    'https://www.google.co.ao/*',
    'https://www.google.com.ar/*',
    'https://www.google.as/*',
    'https://www.google.at/*',
    'https://www.google.com.au/*',
    'https://www.google.az/*',
    'https://www.google.ba/*',
    'https://www.google.com.bd/*',
    'https://www.google.be/*',
    'https://www.google.bf/*',
    'https://www.google.bg/*',
    'https://www.google.com.bh/*',
    'https://www.google.bi/*',
    'https://www.google.bj/*',
    'https://www.google.com.bn/*',
    'https://www.google.com.bo/*',
    'https://www.google.com.br/*',
    'https://www.google.bs/*',
    'https://www.google.bt/*',
    'https://www.google.co.bw/*',
    'https://www.google.by/*',
    'https://www.google.com.bz/*',
    'https://www.google.ca/*',
    'https://www.google.com.kh/*',
    'https://www.google.cc/*',
    'https://www.google.cd/*',
    'https://www.google.cf/*',
    'https://www.google.cat/*',
    'https://www.google.cg/*',
    'https://www.google.ch/*',
    'https://www.google.ci/*',
    'https://www.google.co.ck/*',
    'https://www.google.cl/*',
    'https://www.google.cm/*',
    'https://www.google.cn/*',
    'https://www.google.com.co/*',
    'https://www.google.co.cr/*',
    'https://www.google.com.cu/*',
    'https://www.google.cv/*',
    'https://www.google.com.cy/*',
    'https://www.google.cz/*',
    'https://www.google.de/*',
    'https://www.google.dj/*',
    'https://www.google.dk/*',
    'https://www.google.dm/*',
    'https://www.google.com.do/*',
    'https://www.google.dz/*',
    'https://www.google.com.ec/*',
    'https://www.google.ee/*',
    'https://www.google.com.eg/*',
    'https://www.google.es/*',
    'https://www.google.com.et/*',
    'https://www.google.fi/*',
    'https://www.google.com.fj/*',
    'https://www.google.fm/*',
    'https://www.google.fr/*',
    'https://www.google.ga/*',
    'https://www.google.ge/*',
    'https://www.google.gf/*',
    'https://www.google.gg/*',
    'https://www.google.com.gh/*',
    'https://www.google.com.gi/*',
    'https://www.google.gl/*',
    'https://www.google.gm/*',
    'https://www.google.gp/*',
    'https://www.google.gr/*',
    'https://www.google.com.gt/*',
    'https://www.google.gy/*',
    'https://www.google.com.hk/*',
    'https://www.google.hn/*',
    'https://www.google.hr/*',
    'https://www.google.ht/*',
    'https://www.google.hu/*',
    'https://www.google.co.id/*',
    'https://www.google.iq/*',
    'https://www.google.ie/*',
    'https://www.google.co.il/*',
    'https://www.google.im/*',
    'https://www.google.co.in/*',
    'https://www.google.io/*',
    'https://www.google.is/*',
    'https://www.google.it/*',
    'https://www.google.je/*',
    'https://www.google.com.jm/*',
    'https://www.google.jo/*',
    'https://www.google.co.jp/*',
    'https://www.google.co.ke/*',
    'https://www.google.ki/*',
    'https://www.google.kg/*',
    'https://www.google.co.kr/*',
    'https://www.google.com.kw/*',
    'https://www.google.kz/*',
    'https://www.google.la/*',
    'https://www.google.com.lb/*',
    'https://www.google.com.lc/*',
    'https://www.google.li/*',
    'https://www.google.lk/*',
    'https://www.google.co.ls/*',
    'https://www.google.lt/*',
    'https://www.google.lu/*',
    'https://www.google.lv/*',
    'https://www.google.com.ly/*',
    'https://www.google.co.ma/*',
    'https://www.google.md/*',
    'https://www.google.me/*',
    'https://www.google.mg/*',
    'https://www.google.mk/*',
    'https://www.google.ml/*',
    'https://www.google.com.mm/*',
    'https://www.google.mn/*',
    'https://www.google.ms/*',
    'https://www.google.com.mt/*',
    'https://www.google.mu/*',
    'https://www.google.mv/*',
    'https://www.google.mw/*',
    'https://www.google.com.mx/*',
    'https://www.google.com.my/*',
    'https://www.google.co.mz/*',
    'https://www.google.com.na/*',
    'https://www.google.ne/*',
    'https://www.google.com.nf/*',
    'https://www.google.com.ng/*',
    'https://www.google.com.ni/*',
    'https://www.google.nl/*',
    'https://www.google.no/*',
    'https://www.google.com.np/*',
    'https://www.google.nr/*',
    'https://www.google.nu/*',
    'https://www.google.co.nz/*',
    'https://www.google.com.om/*',
    'https://www.google.com.pk/*',
    'https://www.google.com.pa/*',
    'https://www.google.com.pe/*',
    'https://www.google.com.ph/*',
    'https://www.google.pl/*',
    'https://www.google.com.pg/*',
    'https://www.google.pn/*',
    'https://www.google.com.pr/*',
    'https://www.google.ps/*',
    'https://www.google.pt/*',
    'https://www.google.com.py/*',
    'https://www.google.com.qa/*',
    'https://www.google.ro/*',
    'https://www.google.rs/*',
    'https://www.google.ru/*',
    'https://www.google.rw/*',
    'https://www.google.com.sa/*',
    'https://www.google.com.sb/*',
    'https://www.google.sc/*',
    'https://www.google.se/*',
    'https://www.google.com.sg/*',
    'https://www.google.sh/*',
    'https://www.google.si/*',
    'https://www.google.sk/*',
    'https://www.google.com.sl/*',
    'https://www.google.sn/*',
    'https://www.google.sm/*',
    'https://www.google.so/*',
    'https://www.google.st/*',
    'https://www.google.sr/*',
    'https://www.google.com.sv/*',
    'https://www.google.td/*',
    'https://www.google.tg/*',
    'https://www.google.co.th/*',
    'https://www.google.com.tj/*',
    'https://www.google.tk/*',
    'https://www.google.tl/*',
    'https://www.google.tm/*',
    'https://www.google.to/*',
    'https://www.google.tn/*',
    'https://www.google.com.tr/*',
    'https://www.google.tt/*',
    'https://www.google.com.tw/*',
    'https://www.google.co.tz/*',
    'https://www.google.com.ua/*',
    'https://www.google.co.ug/*',
    'https://www.google.co.uk/*',
    'https://www.google.com.uy/*',
    'https://www.google.co.uz/*',
    'https://www.google.com.vc/*',
    'https://www.google.co.ve/*',
    'https://www.google.vg/*',
    'https://www.google.co.vi/*',
    'https://www.google.com.vn/*',
    'https://www.google.vu/*',
    'https://www.google.ws/*',
    'https://www.google.co.za/*',
    'https://www.google.co.zm/*',
    'https://www.google.co.zw/*'
];

// State management
let temporaryBusinesses = [];
let googleSearches = [];
let lastUrl = '';
let totalFound = 0;

let user = null;

// Initialize on install (no redirect)
chrome.runtime.onInstalled.addListener((details) => {
    if (details.reason === 'install') {
        console.log('Extension installed successfully');
    }
});

// Keep service worker alive using offscreen document
async function createOffscreenDocument() {
    try {
        await chrome.offscreen.createDocument({
            url: 'offscreen.html',
            reasons: ['BLOBS'],
            justification: 'Keep service worker running'
        });
    } catch (error) {
        // Document may already exist
        console.log('Offscreen document status:', error.message);
    }
}

// Start on runtime startup
chrome.runtime.onStartup.addListener(createOffscreenDocument);
createOffscreenDocument();

// Get user identity
async function getUserIdentity() {
    try {
        chrome.identity.getProfileUserInfo((userInfo) => {
            if (userInfo.email !== '' || userInfo.id !== '') {
                user = userInfo;
                chrome.storage.local.set({ 'LastUserCallData': JSON.stringify(userInfo) });
            }
        });
    } catch (error) {
        console.log('User identity error:', error);
    }
}

getUserIdentity();

// Listen for storage changes
chrome.storage.onChanged.addListener((changes, namespace) => {
    for (let [key, { newValue }] of Object.entries(changes)) {
        try {
            if (key === 'bg_businesses_temp' && newValue) {
                temporaryBusinesses = JSON.parse(newValue);
            }
            if (key === 'LastUserCallData' && newValue) {
                user = JSON.parse(newValue);
            }
        } catch (error) {
            console.error('Storage change error:', error);
        }
    }
});

// Save state before suspension
chrome.runtime.onSuspend.addListener(async () => {
    await chrome.storage.local.set({
        bg_businesses_temp: JSON.stringify(temporaryBusinesses)
    });
});

// Listen for messages from popup
chrome.runtime.onMessage.addListener(async (message, sender, sendResponse) => {
    const { type } = message;

    if (type === 'delete') {
        temporaryBusinesses = [];
        googleSearches = [];
        totalFound = 0;
        await chrome.storage.local.set({ TF: 0 });
        sendResponse({ success: true });
    }

    return true; // Keep message channel open for async response
});

// Intercept Google Maps requests
chrome.webRequest.onCompleted.addListener(
    (details) => {
        const url = details.url;
        console.log('[GME] Request intercepted:', url);

        // Check if it's a Google Maps search request
        if (url.includes('/search?tbm=map') || url.includes('/data=')) {
            console.log('[GME] Google Maps search detected:', url);

            // Avoid processing same URL multiple times
            if (lastUrl === url) {
                console.log('[GME] Duplicate URL, skipping');
                return;
            }

            lastUrl = url;

            // Avoid duplicate processing
            if (!googleSearches.includes(url)) {
                googleSearches.push(url);
                console.log('[GME] Processing new URL, total searches:', googleSearches.length);

                // Fetch and process the response
                fetch(url)
                    .then(response => {
                        console.log('[GME] Fetch response status:', response.status);
                        return response.text();
                    })
                    .then(html => {
                        console.log('[GME] HTML received, length:', html.length);
                        processGoogleMapsResponse(url, html);
                    })
                    .catch(error => {
                        console.error('[GME] Fetch error:', error);
                    });
            } else {
                console.log('[GME] URL already in searches array');
            }
        }
    },
    {
        urls: GOOGLE_MAPS_DOMAINS,
        types: ['xmlhttprequest']
    },
    []
);

// Process Google Maps API response
async function processGoogleMapsResponse(url, html) {
    try {
        let data = null;

        // Parse the response based on format
        if (html.includes('window.APP_INITIALIZATION_STATE')) {
            const startIndex = html.indexOf('window.APP_INITIALIZATION_STATE=') + 32;
            const endIndex = html.indexOf(';window.APP_FLAGS');
            const jsonStr = html.slice(startIndex, endIndex);
            const parsed = JSON.parse(jsonStr);
            data = parsed[3][2];
        } else {
            const jsonStr = html.slice(0, -6);
            const parsed = JSON.parse(jsonStr);
            data = parsed.d;
        }

        if (data) {
            const extractedData = data.substr(4);
            parseBusinessData(extractedData);
        }
    } catch (error) {
        console.error('Processing error:', error);
    }
}

// Parse business data from Google Maps response
function parseBusinessData(dataString) {
    try {
        const data = JSON.parse(dataString);
        let businesses = [];

        // Extract business array
        if (data[64] && data[64].length > 0) {
            businesses = data[64];
        } else if (data[0] && data[0][1] && data[0][1].length > 1) {
            businesses = data[0][1];
        } else if (data[64]) {
            businesses = data[64];
        } else {
            return;
        }

        // Process each business
        for (let i = 0; i < businesses.length; i++) {
            const business = businesses[i];

            if (business.length <= 2 && business[business.length - 1][10]) {
                const rawData = business[business.length - 1];
                const cidString = rawData[10].split(':')[1];
                const cid = hexToDecimal(cidString);

                if (!cid) continue;

                // Extract place_id
                let placeId = '';
                if (rawData[4] && rawData[4][3]) {
                    placeId = rawData[4][3][0].split('=')[1].split('&')[0];
                } else {
                    placeId = rawData[78];
                }

                // Check if already processed
                const existingIndex = temporaryBusinesses.findIndex(b => b.place_id === placeId);
                if (existingIndex !== -1 && temporaryBusinesses[existingIndex].checked) {
                    continue;
                }

                // Fetch additional data from API
                fetchBusinessData(placeId, business, cid);
            }
        }
    } catch (error) {
        console.error('Parse error:', error);
    }
}

// Fetch business data from API (using original logic)
async function fetchBusinessData(placeId, googleData, cid) {
    console.log('[GME] fetchBusinessData called for:', placeId);

    // Check if already exists
    const existingIndex = temporaryBusinesses.findIndex(e => e.place_id === placeId);

    if (existingIndex === -1 || temporaryBusinesses[existingIndex] === undefined) {
        totalFound++;
        await chrome.storage.local.set({ TF: totalFound });
        console.log('[GME] New business found, TF:', totalFound);
    } else if (temporaryBusinesses[existingIndex].checked === true) {
        console.log('[GME] Business already checked, skipping');
        return;
    }

    // Build API URL (exactly like original)
    // Build API URL (using random email)
    const randomEmail = Math.random().toString(36).substring(2, 15) + '@gmail.com';
    const apiUrl = `${API_URL}?place_id=${placeId}&email=${randomEmail}`;
    console.log('[GME] API URL:', apiUrl);
    console.log('[GME] Sending data:', { g: googleData, cid: cid });
    console.log('[GME] Google data length:', googleData ? googleData.length : 0);
    console.log('[GME] CID:', cid);

    // Try with application/json instead of text/plain
    await fetch(apiUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',  // ← 改回 application/json 试试
            'Accept': 'application/json'
        },
        mode: 'cors',
        credentials: 'include',
        body: JSON.stringify({ g: googleData, cid: cid })
    })
        .then(async response => {
            console.log('[GME] ========== API Response Debug ==========');
            console.log('[GME] Response:', response);
            console.log('[GME] Response status:', response.status);
            console.log('[GME] Response statusText:', response.statusText);
            console.log('[GME] Response ok:', response.ok);
            console.log('[GME] Response type:', response.type);
            console.log('[GME] Response url:', response.url);
            console.log('[GME] Response redirected:', response.redirected);

            // Log all headers
            console.log('[GME] Response headers:');
            for (let [key, value] of response.headers.entries()) {
                console.log(`[GME]   ${key}: ${value}`);
            }

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            // Clone response to read it multiple times
            const clonedResponse = response.clone();

            // Try to get response text first for debugging
            try {
                const text = await response.text();
                console.log('[GME] ========== Response Text Debug ==========');
                console.log('[GME] Response text type:', typeof text);
                console.log('[GME] Response text length:', text.length);
                console.log('[GME] Response text (first 1000 chars):', text.substring(0, 1000));
                console.log('[GME] Response text (FULL):', text);
                console.log('[GME] Response text trimmed:', text.trim());
                console.log('[GME] Response text === "200":', text === '200');
                console.log('[GME] Response text === "200\\n":', text === '200\n');

                if (!text || text.trim() === '') {
                    console.error('[GME] Response is empty!');
                    throw new Error('Empty response from API');
                }

                // Check if response is just a number (like "200")
                if (text.trim() === '200' || text.trim() === '0' || /^\d+$/.test(text.trim())) {
                    console.error('[GME] API returned a number instead of JSON:', text);
                    console.error('[GME] This might be a server-side error or wrong Content-Type');
                    throw new Error(`API returned invalid response: "${text}"`);
                }

                try {
                    const jsonData = JSON.parse(text);
                    console.log('[GME] ========== JSON Parse Success ==========');
                    console.log('[GME] Parsed data type:', typeof jsonData);
                    console.log('[GME] Parsed data:', jsonData);

                    if (typeof jsonData !== 'object' || jsonData === null) {
                        console.error('[GME] JSON parsed but not an object:', jsonData);
                        throw new Error(`JSON is not an object: ${JSON.stringify(jsonData)}`);
                    }

                    console.log('[GME] JSON keys:', Object.keys(jsonData));
                    return jsonData;
                } catch (e) {
                    console.error('[GME] ========== JSON Parse Failed ==========');
                    console.error('[GME] JSON parse error:', e);
                    console.error('[GME] Error message:', e.message);
                    console.error('[GME] Failed text:', text);
                    console.error('[GME] Failed text (escaped):', JSON.stringify(text));
                    throw new Error(`Invalid JSON: ${e.message}`);
                }
            } catch (e) {
                console.error('[GME] Error reading response text:', e);
                throw e;
            }
        })
        .then(async function (businessData) {
            console.log('[GME] ========== Business Data Received ==========');
            console.log('[GME] Business data type:', typeof businessData);
            console.log('[GME] Business data:', businessData);

            if (!businessData || typeof businessData !== 'object') {
                console.error('[GME] Invalid business data:', businessData);
                throw new Error('Invalid business data received');
            }

            console.log('[GME] Business data keys:', Object.keys(businessData));
            console.log('[GME] Business name:', businessData.name);
            console.log('[GME] Business place_id:', businessData.place_id);

            // Add to storage if not exists
            const index = temporaryBusinesses.findIndex(g => g.place_id === placeId);
            if (index === -1) {
                temporaryBusinesses.push(businessData);
                await chrome.storage.local.set({
                    bg_businesses_temp: JSON.stringify(temporaryBusinesses)
                });
                console.log('[GME] Business added to storage, total:', temporaryBusinesses.length);
            } else {
                console.log('[GME] Business already in storage at index:', index);
            }
        })
        .catch(function (error) {
            console.error('[GME] API fetch error:', error);
            // Store failed URL for retry
            // This line was `await chrome.storage.local.set({ TR: totalRequested });` in the provided snippet,
            // but it's inside a .catch block, so `await` is not directly usable without an `async` function.
            // Given the instruction to make it syntactically correct, I'll remove the `await` here,
            // or wrap it in an async IIFE if the intent was to await.
            // However, the original code had `await chrome.storage.local.set({ TR: totalRequested });`
            // inside the .catch, which is syntactically incorrect.
            // I will keep it as `chrome.storage.local.set` without await to avoid breaking the chain,
            // assuming the intent was to fire-and-forget the storage update in case of error.
            // chrome.storage.local.set({ TR: totalRequested });
        })
        .then(function () {
            // totalRequested++;
            // chrome.storage.local.set({ TR: totalRequested });
            // console.log('[GME] TR updated to:', totalRequested);
        });

    return true;
}

// Base conversion utilities (for CID decoding)
function add(a, b, base) {
    const result = [];
    const maxLength = Math.max(a.length, b.length);
    let carry = 0;

    for (let i = 0; i < maxLength || carry; i++) {
        carry += (i < a.length ? a[i] : 0) + (i < b.length ? b[i] : 0);
        result.push(carry % base);
        carry = Math.floor(carry / base);
    }

    return result;
}

function multiplyByNumber(num, digit, base) {
    if (num < 0) return null;
    if (num === 0) return [];

    let result = [];
    while (true) {
        if (num & 1) {
            result = add(result, digit, base);
        }
        num >>= 1;
        if (num === 0) break;
        digit = add(digit, digit, base);
    }

    return result;
}

function parseToDigitArray(str, fromBase) {
    const chars = str.split('');
    const result = [];

    for (let i = chars.length - 1; i >= 0; i--) {
        const digit = parseInt(chars[i], fromBase);
        if (isNaN(digit)) return null;
        result.push(digit);
    }

    return result;
}

function convertBase(str, fromBase, toBase) {
    const digits = parseToDigitArray(str, fromBase);
    if (!digits) return null;

    let result = [];
    let multiplier = [1];

    for (let i = 0; i < digits.length; i++) {
        if (digits[i]) {
            result = add(result, multiplyByNumber(digits[i], multiplier, toBase), toBase);
        }
        multiplier = multiplyByNumber(fromBase, multiplier, toBase);
    }

    let output = '';
    for (let i = result.length - 1; i >= 0; i--) {
        output += result[i].toString(toBase);
    }

    return output;
}

function hexToDecimal(hex) {
    if (!hex) return null;

    let hexStr = hex;
    if (hex.substring(0, 2) === '0x') {
        hexStr = hex.substring(2);
    } else {
        return null;
    }

    hexStr = hexStr.toLowerCase();
    return convertBase(hexStr, 16, 10);
}
